//==========================================================================
//
// 	ADVMATH.CPP
//
//		Copyright (c) 1996 by James Albertson
//
//==========================================================================

#include <advmath.h>

//--------------------------------------------------------------------------

void send_err_msg(int msg,int id)     /* Display an error message */
{
	printf("\n\nError in Function %s, %s\n\n",funcname[id],error_msg[msg]);
	return;
}

//--------------------------------------------------------------------------

int maxOrdC(int n,double x) /* Order at which to begin recursion
											for cylindrical Bessel fn of 1st kind */
{
	int ord;
	ord = (n>(int)floor(x)) ? n : (int)floor(x);
	return ord+(int)floor(x)+16;
}

double JBesCyl(int n,double x) /* Cylindrical Bessel Fn of the First Kind,
										 A&S 9.12, Example 1 */
{
	int msg=-1,ord;
	double sign,bn,bn1,sum,next,answer;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDCBF1); exit(1); }

	if(x==0 && n==0) return 1;         // special case
	if(x==0 &&  n>0) return 0;         // special case

	bn=1.e-300; bn1=0; sum=0;
	sign = (n&1 && x<0)? -1: 1;
	x=fabs(x);
	ord=maxOrdC(n,x);
	while(ord>=1)
	{
		next=((ord<<1)/x)*bn-bn1;
		if( ord%2 && ord>1 ) sum+=(next+next);
		if(ord==1) sum+=next;           // J0+2J2+2J4+...=1; A&S 9.1.46
		if(n==ord-1) answer=next;
		bn1=bn; bn=next; ord--;
	}
	return sign*answer/sum;
}

//--------------------------------------------------------------------------

double factorial(int n)    /* factorial function */
{
	double fact=n;
	if(n==0 || n==1) return 1;
	while (n>2) fact*=--n;
	return fact;
}

double YBesCyl(int n,double x) /* Cylindrical Bessel Fn of the Second Kind,
												A&S 9.1.88 */
{
	int msg=-1,k;
	double factor=1,coeff1,coeff2,sum,term1,term2,term3;

	if(n<0) msg=IDNGTEQ0;
	if(x<=0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDCBF2); exit(1); }

	if(n==0) term1=0;
	else {
		coeff1=-factorial(n)*pow(0.5*x,-n)/PI;
		coeff2=1;
		sum=JBesCyl(0,x)/n;
		for(k=1;k<n;k++) sum+=(coeff2*=(x/(k<<1)))*JBesCyl(k,x)/(n-k);
		term1=coeff1*sum;
	}

	term2=(log(x)-M_LN2-Digamma(n+1))*JBesCyl(n,x);
	sum=0; k=1;
	do {
		 factor=-factor;
		 term3=(factor*((k<<1)+n)/k/(k+n))*JBesCyl((k<<1)+n,x);
		 sum+=term3;
		 k++;
		 }
	while (fabs(term3/sum)>1.e-18);
	return term1+(2/PI)*(term2-sum);
}

//--------------------------------------------------------------------------

int maxOrdS(int n,double x)   /* Order at which to begin recursion
											for spherical Bessel fn of 1st kind */
{
	int ord;
	ord = (n>(int)floor(x)) ? n : (int)floor(x);
	return ord+(int)(0.5*(n+x))+16;
}

double JBesSph(int n,double x)  /* Sphr. Bessel Fn of the First Kind,
											  A&S 10.5 */
{
	int msg=-1,ord;
	double sign,jn,jn1,term,next,sum,answer;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDSBF1); exit(1); }

	if(n==0 && x==0) return 1;    // special case
	if(n>0  && x==0) return 0;    // special case

	jn=1.e-150; jn1=0; sum=0;
	sign = (n&1 && x<0)? -1 : 1;
	x=fabs(x);
	ord=maxOrdS(n,x);

	while (ord>=0)
	{  term=((ord<<1)+1)*jn;
		sum+=term*jn;    // SUM(2n+1)jn^2=1; A&S 10.1.50
		next=term/x-jn1;
		if(n==(ord-1)) answer=next;
		jn1=jn; jn=next; ord--;
	}
	return sign*answer/sqrt(sum);
}

//--------------------------------------------------------------------------

double YBesSph(int n,double x)   /* Sphr. Bessel Fn of the Second Kind,
												A&S 10.1.12, 10.1.19 */
{
	int msg=-1,ord=2;
	double yn0,yn1,next;
	
	if(n<0) msg=IDNGTEQ0;
	if(x==0) msg=IDXNOTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDSBF2); exit(1); }

	yn0=-cos(x)/x;                // YBesSph(0,x)
	if(n==0) return yn0;
	yn1=yn0/x-sin(x)/x;           // YBesSph(1,x)
	if(n==1) return yn1;

	while (ord<=n)
	{    next=yn1*((ord<<1)-1)/x-yn0;      // YBesSph(2,x)
		yn0=yn1; yn1=next; ord++;
	}
	return yn1;
}

//--------------------------------------------------------------------------

double IBesCyl(int n,double x)   /* Modified Bessel Fn I
												A&S 9.6.10 */
{
	int msg=-1,k;
	double z,term,sum;

	if(n<0 || x<0) msg=IDNXGTEQ0;
	if(x<0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDMBI); exit(1); }

	if(n==0 && x==0) return 1;
	if(n!=0 && x==0) return 0;

	z=0.25*x*x;
	term=1/factorial(n);
	sum=term;
	for(k=1;k<150;k++) {
		term*=z/(k*(n+k));
		sum+=term;
		if(fabs(term/sum)<1.e-18) break;
	}
	return sum*pow(0.5*x,n);
}

//--------------------------------------------------------------------------

double KBesCyl(int n,double x)   /* Modified Bessel Fn K
												A&S 9.6

	>>>>>	RESULTS ARE OF LIMITED PRECISION:  MINIMUM OF 8 PLACES <<<<<	  */

{
	int i,msg=-1;
	double k_0,k_1,k_n;

	if(n<0) msg=IDNGTEQ0;
	if(x<=0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDMBK); exit(1); }

	k_0=k0(x);
	if(n==0) return k_0;

	k_1=(1/x-IBesCyl(1,x)*k_0)/IBesCyl(0,x);  // A&S 9.6.15
	if(n==1) return k_1;

	for(i=1;i<n;i++) {         // A&S 9.6.26, forward recursion on n
		k_n=(2*i/x)*k_1+k_0;
		k_0=k_1;
		k_1=k_n;
	}
	return k_n;
}

double k0(double x)     // x<9
{
	int i;
	double a,z,factor1,factor2,term,sum;

	if(x>=9) return k0_large(x);

	a=-(log(0.5*x)+GAMMA);  // A&S 9.6.13, using 9.6.12 to expand I0(z)
	z=0.25*x*x;
	factor1=a;
	factor2=1;
	sum=a;

	for(i=1;i<50;i++) {
		term=(factor1+=1./i)*(factor2*=z/(i*i));
		sum+=term;
		if(fabs(term/sum)<1.e-18) break;
	}
	return sum;
}

double k0_large(double x)  // x>=9
{
	int i;
	double j,z,term,sum,LIM;

	z=0.25/(x*x);
	sum=1;
	term=1;

	if(x<=10) LIM=pow(10,-8);           // adjust precision
		else if(x<=11) LIM=pow(10,-9);
			else LIM=pow(10,-10);

	for(i=1;i<50;i++) {        // A&S 9.7.5
		j=i<<1;
		term*=(j-1)*(j-1)*(j-1)/j*z;
		sum+=term;
		if(fabs(term/sum)<LIM) break;
	}
	return (0.5/x)*sum/IBesCyl(0,x);
}

//--------------------------------------------------------------------------

double GammaFnc(double x)     /* Gamma Function for x>0, A&S 6.1.15, 6.1.34 */
{
	int msg=-1,i,n;
	double y,power=1,factor=1,sum=0;
	double c[]={ 1.0000000000000000, 0.5772156649015329,-0.6558780715202538,
					-0.0420026350340952, 0.1665386113822915,-0.0421977345555443,
					-0.0096219715278770, 0.0072189432466630,-0.0011651675918591,
					-0.0002152416741149, 0.0001280502823882,-0.0000201348547807,
					-0.0000012504934821, 0.0000011330272320,-0.0000002056338417,
					 0.0000000061160950, 0.0000000050020075,-0.0000000011812746,
					 0.0000000001043427, 0.0000000000077823,-0.0000000000036968,
					 0.0000000000005100,-0.0000000000000206,-0.0000000000000054,
					 0.0000000000000014, 0.0000000000000001  };

	if(x<=0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDGF); exit(1); }

	n=(int)floor(x);        //x=n+y, where n=[x] (largest integer <= x)
	y=x-n;
	for(i=1;i<n;i++) factor*=(x-i);
	if(y==0) return factor;         //factor=(n-1)!=GammaFnc(n)
	for(i=0;i<26;i++) sum+=c[i]*(power*=y); //sum=1/GammaFnc(y), 0<y<1
	factor=(n==0)?factor:factor*y;
	return factor/sum;
}

//-------------------------------------------------------------------------

double psum(int n,double x)   /* auxilliary function for IncGamma,
											 A&S 6.5.13 */
{
	int k;
	double sum=1,term=1;
	for(k=1;k<n;k++)sum+=(term*=x/k);
	return sum*exp(-x);
}

double IncGamma(double a,double x)     /* Incomplete Gamma Function, A&S 6.5 */
{

	int msg=-1;
	double k,denom;

	if(a<0 || x<0) msg=IDAXNLT0;
	if(a==0 && x==0) msg=IDAXNEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDIG); exit(1); }

	if(a==0) return ExpInt(1,x);       //A&S 6.5.15
	if(x==0) return GammaFnc(a);
	if(a==floor(a)) return GammaFnc(a)*psum((int)a,x);    //a = integer

	if(x<=15) return GammaFnc(a)-SmlGamma(a,x);     //A&S 6.5.3
	else {
		denom=x;       //A&S 6.5.31
		for(k=100;k>0;k-=1) denom=x+(k-a)/(1+k/denom);
		return exp(-x)*pow(x,a)/denom;
	}
}

//-------------------------------------------------------------------

double SmlGamma(double a,double x)    /* Small Gamma, A&S 6.5.2, 6.5.3 */
{
	int msg=-1,k;
	double sum,term,factor=1;

	if(a<=0 || x<0) msg=IDAGT0XNLT0;
	if(!(msg<0)) { send_err_msg(msg,IDSG); exit(1); }

	if(x>15) return GammaFnc(a)-IncGamma(a,x);   //A&S 6.5.3
	else {
		sum=1/GammaFnc(a+1);       //A&S 6.5.4,6.5.29
		for(k=1;k<100;k++) {
			term=(factor*=x)/GammaFnc(a+k+1);
			sum+=term;
			if(fabs(term/sum)<1.e-18) break; }
		return exp(-x)*GammaFnc(a)*pow(x,a)*sum;
	}
}

//-------------------------------------------------------------------

double Digamma(double x)   // Digamma or Psi Function, A&S 6.3.1
{
	int msg=-1;

	if(x<=0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDDIG); exit(1); }

	if(x==floor(x)) {		//integer argument, A&S 6.3.2
		double sum=0;
		while (x>1) sum+=1/(--x);
		return -GAMMA+sum;
	}

	return PolyGam(0,x);
}

//-------------------------------------------------------------------

double PolyGam(int n,double x) // Polygamma Function, A&S 6.4.1
										 // Digamma (Psi) Function if n=0, A&S 6.3.1
{
	int msg=-1;

	if(n<0) msg=IDNGTEQ0;
	if(x<=0) msg=IDXGT0;
	if(!(msg<0)) { send_err_msg(msg,IDPOG); exit(1); }

	// can use asymptotic formula if x>=40 for Polygamma, x>=10 for Digamma
	if(x>=(n?40:10)) return polytop(n,x);

	// otherwise use backward recursion on x
	return polydown(n,x);
}

double polytop(int n,double x)   // A&S 6.3.18, 6.4.11, 23.2.16
{
	int k,l;
	double p,factor1,factor2,term,sum=0;

	l=n-(n>0);
	factor1=factorial(l);
	factor2=-1/(4*PI*PI*x*x);
	for(k=1;k<50;k++) {
		p=++l;
		p*=++l;
		term=-Zeta(k<<1)*(factor1*=p*factor2);
		sum+=n?term:term/k;
		if(fabs(term/sum)<1.e-18) break;
	}
	if(n==0) return log(x)-0.5/x-sum;
	return ((n&1)?1:-1)*(factorial(n-1)*(1+0.5*n/x)+2*sum)*pow(x,-n);
}

double polydown(int n,double x)  // A&S 6.3.5, 6.4.6 (backward recursion)
{
	double z,m,poly,factor;

	m=floor(x);    // m = largest integer <= x
	z=x-m;         // z = fractional part of x, 0<=z<1
	x=n?40:10;		// x = 40 for Polygamma, 10 for Digamma

	factor=((n&1)?1:-1)*factorial(n);
	poly=polytop(n,x+z);    // preserve full precision of fractional part
	for(--x;x>=m;x--) poly+=factor*pow(x+z,-n-1);
	return poly;
}

//-------------------------------------------------------------------

double ExpInt(int n,double x)      /* Exponential Integral, A&S 5.1.4 */
{
	int msg=-1,i;
	double e1;

	if(n<0 || x<0) msg=IDNXGTEQ0;
	if(x==0 && n<=1) msg=IDNGT1X0;
	if(!(msg<0)) { send_err_msg(msg,IDEXI); exit(1); }

	if(x==0 && n>1) return 1./(n-1);  //A&S 5.1.23

	if(n==0) return exp(-x)/x;         //A&S 5.1.24

	if(x<1) {
		e1=e1_pwr(x);
		if(n==1) return e1;
		for(i=1;i<=(n-1);i++)
			e1=(1./i)*(exp(-x)-x*e1);     //A&S 5.1.14
		return e1;
	}
	return en_cnf(n,x);
}

double e1_pwr(double x)  //A&S 5.1.11
{
	int i;
	double sum,term,factor=1;

	sum=-GAMMA-log(x);

	for(i=1;i<25;i++) {
		term=(factor*=-x/i)/i;
		sum-=term;
		if(fabs(term/sum)<1.e-18) break;
	}
	return sum;
}

double en_cnf(int n,double x)  //A&S 5.1.22
{
	double a,denom=0;

	for(a=200;a>=1;a-=1) denom=(a-1+n)/(1+a/(x+denom));

	return exp(-x)/(x+denom);
}

//-------------------------------------------------------------------

double Ei(double x)  /* Cauchy Principal Value for -ExpInt(1,-x)
									when x>0, A&S 5.1.2 */
{
	int msg=-1,i;
	double sum,term,factor=1;

	if(x<=0 || x>50) msg=IDXGT0LT50;
	if(!(msg<0)) { send_err_msg(msg,IDEI); exit(1); }

	sum=GAMMA+log(x);             //A&S 5.1.10
	for(i=1;i<125;i++) {
		term=(factor*=x/i)/i;
		sum+=term;
		if(fabs(term/sum)<1.e-18) break;
	}
	return sum;
}

//--------------------------------------------------------------------------

double Chb1Poly(int n,double x) /* Chebyshev polynomial T(n,x),
												A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDCP1); exit(1); }

	p0=1;
	if(n==0) return p0;
	p1=x;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=2*x*p1-p0;
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double Chb2Poly(int n,double x) /* Chebyshev polynomial U(n,x),
												A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDCP2); exit(1); }

	p0=1;
	if(n==0) return p0;
	p1=2*x;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=2*x*p1-p0;
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double GegPoly(double a,int n,double x) /* Gegenbauer (Ultraspherical)
															polynomial, A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDGP); exit(1); }

	if(a==0) {
		if(n==0) return 1;           //A&S 22.4.3
		return 2*Chb1Poly(n,x)/n;    //A&S 22.5.33
	}

	p0=1;
	if(n==0) return p0;
	p1=2*a*x;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=(2*(i+a)*x*p1-(i+2*a-1)*p0)/(i+1);
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double HerPoly(int n,double x)  /* Hermite polynomial, A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDHP); exit(1); }

	p0=1;
	if(n==0) return p0;
	p1=2*x;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=2*x*p1-(i<<1)*p0;
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double JacPoly(double a,double b,int n,double x)   /* Jacobi polynomial,
																			A&S p.789 */
{
	int msg=-1,i;
	double am=1,dn;

	if(n<0) msg=IDNGTEQ0;
	if(a<=-1) msg=IDAGTMIN1;
	if(!(msg<0)) { send_err_msg(msg,IDJP); exit(1); }

	if(n==0) return 1;
	dn=GammaFnc(n+a+1)/factorial(n)/GammaFnc(a+1);
	for(i=n;i>0;i--) am=1-(n-i+1)*(a+b+n+i)/((i<<1)*(a+i))*(1-x)*am;
	return dn*am;
}

//--------------------------------------------------------------------------

double LagPoly(double a,int n,double x) /* Generalized Laguerre polynomial,
															A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDGLP); exit(1); }

	p0=1;
	if(n==0) return p0;
	p1=1-x+a;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=(((i<<1)+a+1-x)*p1-(i+a)*p0)/(i+1);
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double LegPoly(int n,double x)  /* Legendre polynomial, A&S 22.4, 22.7 */
{
	int msg=-1,i;
	double p0,p1,p2;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDLP); exit(1); }

	p0=1;
	if(n==0 || x==1) return p0;
	p1=x;
	if(n==1) return p1;
	for(i=1;i<n;i++)
	{
		p2=(((i<<1)+1)*x*p1-i*p0)/(i+1);
		p0=p1;
		p1=p2;
	}
	return p2;
}

//--------------------------------------------------------------------------

double AscLeg(int l,int m,double x)        /* Associated Legendre function,
																A&S 8.5.3 */
{
	int msg=-1,i;
	double pmm,pm1,pm2,coef,factor;

	if(l<0) msg=IDLGTEQ0;
	if(l>=0 && (m<0 || m>l) ) msg=IDM0TOL;
	if(fabs(x)>1) msg=IDABSXLT1;
	if(!(msg<0)) { send_err_msg(msg,IDALF); exit(1); }

	pmm=1;                  // P(m,m,x)=(-1)^m(2m-1)!!(1-x^2)^(m/2)
	coef=sqrt((1-x)*(1+x));
	factor=1;
	for(i=1;i<=m;i++) {     // for a different sign convention
		pmm*=-factor*coef;   //   use:  pmm*=factor*coef;
		factor+=2;
	}

	if(l==m) return pmm;    // P(m,m,x)

	pm1=x*((m<<1)+1)*pmm;   // P(m+1,m,x) [P(m-1,m,x)=0]

	if(l==(m+1)) return pm1;

	for(i=m+2;i<=l;i++) {
		pm2=(((i<<1)-1)*x*pm1-(i+m-1)*pmm)/(i-m); // P(m+2,m,x)
		pmm=pm1;
		pm1=pm2;
	}

	return pm2;
}

//-------------------------------------------------------------------------

double Error(double x)            /* Error Function, A&S 7.1.1 */
{
	int sign=1;

	if(x<0)sign=0;
	x=fabs(x);
	if(x<=1) return (sign)? erfpwr(x) : -erfpwr(x);
	else return (sign)? erfcnf(x) : -erfcnf(x);
}

double erfpwr(double x)         //A&S 7.1.5
{
	int i;
	double sum,term;

	if(x==0) return 0;
	sum=x; term=x;
	for(i=1;i<25;i++) {
		term*=-x*x/i;
		sum+=term/((i<<1)+1);
		if(fabs(term/sum)<1.e-18) break;
	}
	return sum*2/sqrt(PI);
}

double erfcnf(double x)         //A&S 7.1.14
{
	double a,denom;

	denom=x;
	for(a=100;a>0;a-=0.5) denom=x+a/denom;
	return 1-exp(-x*x)/(sqrt(PI)*denom);
}

//-------------------------------------------------------------------------

double Nprob(double x)  /* Normal (Gaussian) Probability Function,
										A&S 26.2.2 */
{
	int sign=1;

	if(x<0)sign=0;
	x=fabs(x);

	if(x<1.3)return (sign)? p_pwr(x) : 1-p_pwr(x);
	else return (sign)? 1-q_cnf(x) : q_cnf(x);
}

double p_pwr(double x)          //A&S 26.2.11
{
	int i;
	double sum,term;

	if(x==0) return 0.5;
	sum=x; term=x;
	for(i=1;i<25;i++) {
		term*=x*x/((i<<1)+1);
		sum+=term;
		if(fabs(term/sum)<1.e-18) break;
	}
	return 0.5+sum*exp(-0.5*x*x)/sqrt(2*PI);
}

double q_cnf(double x)          //A&S 26.2.14
{
	double a,denom;

	denom=x;
	for(a=200;a>0;a-=1) denom=x+a/denom;
	return exp(-0.5*x*x)/(sqrt(2*PI)*denom);
}

//-------------------------------------------------------------------------

double ChiSqrQ(double Xsq,int nu)    /* Chi-Square Probability Function,
														A&S 26.4.2, 26.4.19 */
{
	int msg=-1;

	if(Xsq<0) msg=IDXSQGT0;
	if(nu<1) msg=IDNUGTEQ1;
	if(!(msg<0)) { send_err_msg(msg,IDXSQQ); exit(1); }

	return IncGamma(0.5*nu,0.5*Xsq)/GammaFnc(0.5*nu);
}

//-------------------------------------------------------------------

double bicoef(int n,int k)    /* binomial coefficient, A&S 3.1.2 */
{
	return factorial(n)/(factorial(k)*factorial(n-k));
}

double power(double x,int n)  /* use in BernPoly and EulerPoly to avoid a
											domain error in calling pow(0,0) */
{
	if(x==0 && n==0) return 1;
	return pow(x,n);
}

double BernNum(int n)   /* Bernoulli Number, A&S 23.2.16 */
{
	int msg=-1;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDBN); exit(1); }

	if(n==0) return 1;
	if(n==1) return -0.5;
	if(n&1) return 0;
	return (((n>>1)&1)?2:-2)*factorial(n)*pow(2*PI,-n)*Zeta(n);
}

double BernPoly(int n,double x)  /* Bernoulli Polynomial; A&S 23.1.20,23.1.7 */
{
	int msg=-1,k;
	double b_poly=0;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDBP); exit(1); }

	for(k=0;k<=n;k++) b_poly+=bicoef(n,k)*BernNum(k)*power(x,n-k);
	return b_poly;
}

double EulerNum(int n)  /* Euler Number; A&S 23.1.2, 23.1.19 */
{
	int msg=-1;

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDEN); exit(1); }

	if(n==0) return 1;
	if(n&1) return 0;
	return pow(2,n)*EulerPoly(n,.5);
}

double EulerPoly(int n,double x) /* Euler Polynomial; A&S 23.1.20,23.1.7 */
{
	int msg=-1,k;
	double e_poly=power(x,n);

	if(n<0) msg=IDNGTEQ0;
	if(!(msg<0)) { send_err_msg(msg,IDEP); exit(1); }

	for(k=1;k<=n;k++) e_poly+=bicoef(n,k)*2./(k+1)*(1-pow(2,k+1))\
		*BernNum(k+1)*power(x,n-k);
	return e_poly;
}

//-------------------------------------------------------------------

double Zeta(int n)       /* Riemann Zeta Function, A&S 23.2.1 */
{
	int msg=-1,k;
	double value[]= { 1.64493406684822644, 1.20205690315959429,
							1.08232323371113819, 1.03692775514336993,
							1.01734306198444914, 1.00834927738192283,
							1.00407735619794434, 1.00200839282608221,
							1.00099457512781809, 1.00049418860411946 },
		term,zeta=0;

	if(n<2) msg=IDNGT1;
	if(!(msg<0)) { send_err_msg(msg,IDZETA); exit(1); }

	if(n>57) return 1;
	if(n<12) return value[n-2];
	for(k=1;k<30;k++) {
		term=pow(k,-n);
		zeta+=term;
		if(fabs(term/zeta)<1.e-18) break; }
	return zeta;
}

//-------------------------------------------------------------------------

double Adet(double* array,int order)
{
	return find_det(array,order);
}

// move column with largest absolute value in first row into first column
void move_largest(double *dummy,int size,int order,double &sign)
{
	int i,iorder,col;
	double largest;

	// find col of element with largest abs value in first row
	largest=fabs(dummy[0]);
	col=0;
	for(i=1;i<size;i++)
		if(fabs(dummy[i])>largest) { largest=fabs(dummy[i]); col=i; }

	// move that column into first position
	if(col!=0) {
		for(i=0;i<size;i++) {
			iorder=i*order;
			largest=dummy[iorder];
			dummy[iorder]=dummy[iorder+col];
			dummy[iorder+col]=largest;
		}
		sign=-sign;
	}
	return;
}

// find determinant of matrix
double find_det(double *dummy,int order)
{
	const double LIM = 1.e-12;
	//for "approximately" proportional rows & cols in singular matrix

	typedef unsigned long ul;

	double far* col;
	double factor=1,sign=1,ratio,ratio1,ratio2;
	int i,j,k,msg,size,singular,first,line,iline,jline;

	// look for duplicate columns (singular matrix)
	singular=0;
	for(i=0;i<(order-1);i++) {
		for(j=i+1;j<order;j++) {
			if(dummy[i]==dummy[j]) { // check cols. if dup. in 1st row
				singular=1;
				for(k=1;k<order;k++) {
					if(dummy[k*order+i]!=dummy[k*order+j])
						{singular=0; break;}
				}
			}
		}
	}
	if(singular==1) return 0;

	// look for duplicate rows (singular matrix)
	singular=0;
	for(i=0;i<(order-1);i++) {
		iline=i*order;
		for(j=i+1;j<order;j++) {
			jline=j*order;
			if(dummy[iline]==dummy[jline]) { // check rows if dup in 1st col
				singular=1;
				for(k=1;k<order;k++) {
					if(dummy[iline+k]!=dummy[jline+k])
						{singular=0; break;}
				}
			}
		}
	}
	if(singular==1) return 0;

	// look for proportional columns (singular matrix)
	for(i=0;i<(order-1);i++) {
		for(j=i+1;j<order;j++) {
			singular=1; first=-1;
			for(k=0;k<order;k++) {
				line=k*order;
				if( (dummy[line+i]==0) ^ (dummy[line+j]==0) )
					{singular=0; break;}
				if( (dummy[line+i]==0) & (dummy[line+j]==0) ) continue;
				if(first+1==0) { first=k; ratio1=dummy[line+i]/dummy[line+j]; }
				if(first>=0) { ratio2=dummy[line+i]/dummy[line+j];
					ratio=ratio1/ratio2;
					if( ratio<0 || ( ratio<1-LIM || ratio>1+LIM ) )
					{singular=0; break;}
				}
			}
			if(singular==0) continue;
			return 0;
		}
	}

	// look for proportional rows (singular matrix)
	for(i=0;i<(order-1);i++) {
		iline=i*order;
		for(j=i+1;j<order;j++) {
			singular=1; first=-1;
			jline=j*order;
			for(k=0;k<order;k++) {
				if( (dummy[iline+k]==0) ^ (dummy[jline+k]==0) )
					{singular=0; break;}
				if( (dummy[iline+k]==0) & (dummy[jline+k]==0) ) continue;
				if(first+1==0) { first=k; ratio1=dummy[iline+k]/dummy[jline+k]; }
				if(first>=0) { ratio2=dummy[iline+k]/dummy[jline+k];
					ratio=ratio1/ratio2;
					if( ratio<0 || ( ratio<1-LIM || ratio>1+LIM ) )
					{singular=0; break;}
				}
			}
			if(singular==0) continue;
			return 0;
		}
	}

	// allocate memory for col
	if((col=(double far*)farcalloc((ul)order,(ul)sizeof(double)))==NULL) {
		send_err_msg(IDALLOC,IDADET); exit(1);
	}

	// evaluate determinant of non-singular matrix by pivotal condensation
	for(size=order;size>0;size--) {

		move_largest(dummy,size,order,sign);

		// divide first row by leading element
		factor*=dummy[0];
		if(dummy[0]!=0) {
			for(i=1;i<size;i++)
				dummy[i]/=dummy[0];
			dummy[0]=1;
		}
		else break;

		// form vector of first col
		for(i=0;i<size;i++) col[i]=dummy[i*order];

		// pivot
		for(j=0;j<(size-1);j++) {
			for(i=0;i<(size-1);i++)
			dummy[i*order+j]=dummy[(i+1)*order+(j+1)]-col[i+1]*dummy[j+1];
		}
	}

	farfree(col);
	return factor*sign;
}

//-------------------------------------------------------------------------
